/**
 * \file: mspin_udp_unicast.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN UDP Unicast Support
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
 * \copyright: (c) 2018 Bosch SoftTec GmbH
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_udp_unicast.h"
#include "mspin_udp_helper.h"
#include "mspin_logging.h"

#include <errno.h>          //errno
#include <sys/prctl.h>      //prctl, PR_SET_NAME
#include <arpa/inet.h>      //inet_ntop

typedef struct
{
    mspin_udp_MessageParameter_t message;
    in_addr_t ipAddr;
    in_addr_t netmask;
    in_addr_t destIPAddr;
    BOOL quitUDPUnicasting;
    pthread_t unicastThreadID;
    MSPIN_OnUDPUnicastEnd onUDPUnicastEnd;
    void* onUDPUnicastContext;
    pthread_mutex_t performUnicastLock;
    pthread_cond_t abortUnicast;
} mspin_udp_UnicastContext_t;

mspin_udp_UnicastContext_t *gpUDPUnicastContext = NULL;

static void mspin_udp_freeUnicastContext(mspin_udp_UnicastContext_t** ppContext)
{
    //Frees only memory. Does not destroy any pthread_* variables
    mspin_udp_UnicastContext_t* pContext = *ppContext;

    if (pContext)
    {
        mspin_udp_freeBTName(&(pContext->message));

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) deleting context...",
                __FUNCTION__, pContext);

        free(pContext);
        *ppContext = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) context deleted",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(context=%p) WARNING: UDP broadcast context is NULL, nothing to do",
                __FUNCTION__, pContext);
    }
}

static mspin_udp_UnicastContext_t* mspin_udp_createUnicastContext(void)
{
    mspin_udp_UnicastContext_t* pContext = NULL;
    int rc = -1;

    pContext = malloc(sizeof(mspin_udp_UnicastContext_t));

    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate memory for broadcast context",
                __FUNCTION__);
        return NULL;
    }

    memset(pContext, 0, sizeof(mspin_udp_UnicastContext_t));

    //Initialize broadcast mutex
    rc = pthread_mutex_init(&(pContext->performUnicastLock), NULL); //use default mutex attributes
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize broadcast mutex with %d",
                __FUNCTION__, rc);

        mspin_udp_freeUnicastContext(&pContext); //deletes only memory
        return NULL;
    }

    //Initialize broadcast condition variable
    rc = pthread_cond_init(&(pContext->abortUnicast), NULL); //use default condition attributes
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize broadcast mutex with %d",
                __FUNCTION__, rc);

        pthread_mutex_destroy(&(pContext->performUnicastLock));
        mspin_udp_freeUnicastContext(&pContext); //deletes only memory
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s() UDP broadcast context=%p created",
            __FUNCTION__, pContext);

    return pContext;
}

static void mspin_udp_deleteUnicastContext(mspin_udp_UnicastContext_t** ppContext)
{
    //Destroys the pthread_* variables and then frees the memory
    mspin_udp_UnicastContext_t* pContext = *ppContext;
    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered", __FUNCTION__, pContext);

        pthread_mutex_destroy(&(pContext->performUnicastLock));
        pthread_cond_destroy(&(pContext->abortUnicast));

        mspin_udp_freeUnicastContext(ppContext);

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) finished", __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(*ctx=%p) FATAL ERROR: pContext is NULL",
                __FUNCTION__, ppContext);
    }
}

static void* mspin_udp_unicastThread(void* exinf)
{
    int sockfd = -1;
    int errorCount = 0;
    char hostString[INET_ADDRSTRLEN] = "";
    char targetAddr[INET_ADDRSTRLEN] = "";
    char udpMessage[MSPIN_UDP_PACKET_MAX_LENGTH] = "";
    struct timespec ts = {0};
    struct sockaddr_in unicastAddr = {0};
    MSPIN_UDP_BROADCAST_END_REASON endReason = MSPIN_UDP_BROADCAST_END_REASON_QUIT;

    //Check context
    mspin_udp_UnicastContext_t *pContext = (mspin_udp_UnicastContext_t*)exinf;
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(exinf=%p) FATAL ERROR: Context is NULL",
                __FUNCTION__, exinf);

        //Callback can't be issued because context is not valid

        pthread_exit(NULL);
        return NULL;
    }

    //Set thread name
    //                  123456789 123456
    prctl(PR_SET_NAME, "mspin_unicstUDP", 0, 0, 0);

    //Unicast address
    inet_ntop(AF_INET, &pContext->ipAddr, hostString, INET_ADDRSTRLEN);

    //Target IP address
    inet_ntop(AF_INET, &pContext->destIPAddr, targetAddr, INET_ADDRSTRLEN);

    //Create socket
    sockfd = socket(AF_INET, SOCK_DGRAM, 0);
    if (-1 == sockfd)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: Failed to create socket with '%s'(%d)",
                __FUNCTION__, exinf, strerror(errno), errno);

        close(sockfd);

        if (pContext->onUDPUnicastEnd)
        {
            pContext->onUDPUnicastEnd(MSPIN_UDP_BROADCAST_END_REASON_SETUP_FAILED, pContext->onUDPUnicastContext);
        }

        pthread_exit(exinf);
        return NULL;
    }

    //Construct local address structure
    memset(&unicastAddr, 0, sizeof(unicastAddr));
    unicastAddr.sin_family = AF_INET;
    unicastAddr.sin_port = htons(pContext->message.udpPort);  // short, network byte order
    unicastAddr.sin_addr.s_addr = pContext->destIPAddr;

    //Create UDP message
    if (*hostString)
    {
        if (0 != mspin_udp_createUDPMessage(&(pContext->message), udpMessage, sizeof(udpMessage), hostString))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: UDP message too long -> abort",
                    __FUNCTION__, exinf);

            close(sockfd);

            if (pContext->onUDPUnicastEnd)
            {
                pContext->onUDPUnicastEnd(MSPIN_UDP_BROADCAST_END_REASON_IP_ADDRESS_FAILURE,
                        pContext->onUDPUnicastContext);
            }

            pthread_exit(exinf);
            return NULL;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) message to be sent is: '%s'",
                    __FUNCTION__, exinf, udpMessage);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: host address is empty",
                __FUNCTION__, exinf);

        close(sockfd);

        if (pContext->onUDPUnicastEnd)
        {
            pContext->onUDPUnicastEnd(MSPIN_UDP_BROADCAST_END_REASON_IP_ADDRESS_FAILURE,
                    pContext->onUDPUnicastContext);
        }

        pthread_exit(exinf);
        return NULL;
    }

    pthread_mutex_lock(&(pContext->performUnicastLock));

    /* PRQA: Lint Message 64: accept type mismatch */
    /*lint -save -e64*/
    //Start unicasting
    while(!pContext->quitUDPUnicasting)
    {
        if (-1 == sendto(sockfd, udpMessage, strlen(udpMessage), 0, (struct sockaddr *)&unicastAddr, sizeof(unicastAddr)))
        {
            if (++errorCount >= 3)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(exinf=%p) ERROR: Failed %d to send UDP unicast message with '%s'(%d)",
                        __FUNCTION__, exinf, errorCount, strerror(errno), errno);

                endReason = MSPIN_UDP_BROADCAST_END_REASON_SEND_FAILED;
                pContext->quitUDPUnicasting = TRUE;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityWarn,
                        "%s(exinf=%p) WARNING: Failed %d to send UDP unicast message with '%s'(%d)",
                        __FUNCTION__, exinf, errorCount, strerror(errno), errno);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) UDP packet sent to '%s' with mesg='%s'",
                    __FUNCTION__, exinf, targetAddr, udpMessage);
        }

        //Wait till timeout or end condition. First set timer
        clock_gettime(CLOCK_REALTIME, &ts);
        ts.tv_sec += pContext->message.timeout/1000;
        ts.tv_nsec += (pContext->message.timeout % 1000) * 1000000;
        if (ts.tv_nsec >= 1000000000L)
        {
            ts.tv_sec++;
            ts.tv_nsec -= 1000000000L;
        }

        pthread_cond_timedwait(&(pContext->abortUnicast), &(pContext->performUnicastLock), &ts);
    }
    /*lint -restore*/

    pthread_mutex_unlock(&(pContext->performUnicastLock));

    //Close socket
    close(sockfd);

    if (pContext->onUDPUnicastEnd)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) issue UDP unicast end CB", __FUNCTION__, exinf);
        pContext->onUDPUnicastEnd(endReason, pContext->onUDPUnicastContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(exinf=%p) WARNING: No UDP unicast end callback registered",
                __FUNCTION__, exinf);
    }

    // if not stopped from outside, we have to clean up
    if (endReason == MSPIN_UDP_BROADCAST_END_REASON_SEND_FAILED)
    {
        mspin_udp_deleteUnicastContext(&gpUDPUnicastContext);
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) terminate thread", __FUNCTION__, exinf);

    pthread_exit(exinf);
    return NULL;
}

static S32 mspin_udp_startUnicastingThread(mspin_udp_UnicastContext_t* pContext)
{
    pthread_attr_t attr;
    S32 rc = -1;

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return rc;
    }

    pContext->quitUDPUnicasting = FALSE;

    rc = pthread_create(&pContext->unicastThreadID, &attr, mspin_udp_unicastThread, (void*)pContext);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, pContext->unicastThreadID);
    }

    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Setting name failed with rc=%d", __FUNCTION__, rc);
        pContext->unicastThreadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

static void mspin_udp_stopUDPUnicastThread(mspin_udp_UnicastContext_t* pContext)
{
    void* status = NULL;

    if (pContext && (0 != pContext->unicastThreadID))
    {
        pthread_t threadID = pContext->unicastThreadID;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) quit and wake up thread=%lu",
                __FUNCTION__, pContext, threadID);

        pthread_mutex_lock(&(pContext->performUnicastLock));
        pContext->quitUDPUnicasting = true;
        pthread_cond_signal(&(pContext->abortUnicast));
        pthread_mutex_unlock(&(pContext->performUnicastLock));

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) join thread=%lu", __FUNCTION__, pContext, threadID);

        (void)pthread_join(pContext->unicastThreadID, &status);
        pContext->unicastThreadID = 0;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) thread=%lu joined", __FUNCTION__, pContext, threadID);
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p) FATAL ERROR: Context handle is NULL",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(ctx=%p) WARNING: Broadcast thread ID is 0",
                __FUNCTION__, pContext);
    }
}

MSPIN_ERROR mspin_udp_startUDPUnicasting(S32 udpPort, const U8* interface,
        const char* destIPAddr, MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters,
        U32 timeout, MSPIN_OnUDPUnicastEnd onUDPUnicastEnd,
        void* onUDPUnicastEndContext)
{
    MSPIN_ERROR result = mspin_udp_validateMessageParameters(&messageParameters);
    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(udpPort=%d, interface='%s') ERROR: Failed to validate message parameters with %s",
               __FUNCTION__, udpPort, interface, MSPIN_GetErrorName(result));
        return result;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(udpPort=%d, interface='%s', destIPAddr='%s', timeout=%dms, cb=%p, context=%p) entered",
            __FUNCTION__, udpPort, interface, (const char*)destIPAddr, timeout, onUDPUnicastEnd, onUDPUnicastEndContext);

    //Initialize server context
    if (!gpUDPUnicastContext)
    {
        gpUDPUnicastContext = mspin_udp_createUnicastContext();

        if (!gpUDPUnicastContext)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(udpPort=%d, interface='%s') ERROR: Failed to create unicast context",
                   __FUNCTION__, udpPort, interface);
            return MSPIN_ERROR_GENERAL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(udpPort=%d, interface='%s') WARNING: UDP unicast context is already present -> return ALREADY_RUNNING",
                __FUNCTION__, udpPort, interface);
        return MSPIN_ERROR_ALREADY_RUNNING;
    }

    //Store and reset parameters
    gpUDPUnicastContext->message.udpPort = udpPort;
    gpUDPUnicastContext->message.timeout = timeout;
    gpUDPUnicastContext->onUDPUnicastEnd = onUDPUnicastEnd;
    gpUDPUnicastContext->onUDPUnicastContext = onUDPUnicastEndContext;
    gpUDPUnicastContext->quitUDPUnicasting = FALSE;

    gpUDPUnicastContext->destIPAddr = inet_addr(destIPAddr);
    if (INADDR_NONE == gpUDPUnicastContext->destIPAddr)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(udpPort=%d, interface='%s') ERROR: Failed to convert IP address into binary data in network order",
                __FUNCTION__, udpPort, interface);
        return MSPIN_ERROR_INVALID_PARAMETER;
    }

    //Copy message parameters
    result = mspin_udp_copyMessageParameters(&messageParameters, &(gpUDPUnicastContext->message));
    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(udpPort=%d, interface='%s') ERROR: Failed to copy message parameters -> delete context again",
                __FUNCTION__, udpPort, interface);
        mspin_udp_deleteUnicastContext(&gpUDPUnicastContext);
    }

    //Start UDP unicasting
    if (MSPIN_SUCCESS == result)
    {
        //Get local IP address for specified interface
        result = mspin_udp_getBroadcastAddress(interface, &(gpUDPUnicastContext->ipAddr), &(gpUDPUnicastContext->netmask));
        if (MSPIN_SUCCESS == result)
        {
            //Start UDP unicasting thread
            if (0 != mspin_udp_startUnicastingThread(gpUDPUnicastContext))
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(udpPort=%d, interface='%s') ERROR: Failed to start unicasting thread -> delete context again",
                        __FUNCTION__, udpPort, interface);

                mspin_udp_deleteUnicastContext(&gpUDPUnicastContext);

                result = MSPIN_ERROR_GENERAL;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(udpPort=%d, interface='%s') ERROR: Failed to get local IP address -> delete context again",
                    __FUNCTION__, udpPort, interface);

            mspin_udp_deleteUnicastContext(&gpUDPUnicastContext);
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(udpPort=%d, interface='%s') returns '%s'",
            __FUNCTION__, udpPort, interface, MSPIN_GetErrorName(result));

    return result;
}

MSPIN_ERROR mspin_udp_stopUDPUnicasting(void)
{
    if (gpUDPUnicastContext)
    {
        //Check if called from UDP broadcast thread
        if (gpUDPUnicastContext->unicastThreadID == pthread_self())
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Not allowed to call from unicast thread! For example when issued from CB 'onUDPUnicastEnd'",
                    __FUNCTION__);
            return MSPIN_ERROR_NOT_ALLOWED;
        }

        mspin_udp_stopUDPUnicastThread(gpUDPUnicastContext);

        //Reset values
        gpUDPUnicastContext->message.udpPort = 0;
        gpUDPUnicastContext->message.tcpPort = 0;
        gpUDPUnicastContext->message.timeout = 0;
        gpUDPUnicastContext->onUDPUnicastEnd = NULL;
        gpUDPUnicastContext->onUDPUnicastContext = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s() thread joined => delete context", __FUNCTION__);

        mspin_udp_deleteUnicastContext(&gpUDPUnicastContext);

        mspin_log_printLn(eMspinVerbosityDebug, "%s() done", __FUNCTION__);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: No UDP unicast context found", __FUNCTION__);
    }

    return MSPIN_SUCCESS;
}

